#ifndef __KRAKATOAAPI__
#define __KRAKATOAAPI__

#ifdef WIN32
#define FCNEXPORT extern "C" __declspec(dllexport)
#define CLSEXPORT __declspec(dllexport)
#else
#define FCNEXPORT TODO//
#define CLSEXPORT TODO//
#endif

namespace krakatoasr {

#ifdef WIN32
typedef __int64 INT64;
#else
typedef long long INT64;
#endif

enum logging_level_t {
	LOG_NONE = 0,
	LOG_ERRORS,
	LOG_WARNINGS,
	LOG_PROGRESS,
	LOG_STATS,
	LOG_DEBUG,
	LOG_CUSTOM = -1
};

enum output_types_t {
	OUTPUT_RGBA,
	OUTPUT_Z,
	OUTPUT_NORMAL
};

enum rendering_method_t {
	METHOD_PARTICLE,
	METHOD_VOXEL
};

enum filter_t {
	FILTER_BILINEAR, //when using bilinear, also specify a filter "size" (default to 1)
	FILTER_BICUBIC,
	FILTER_NEAREST_NEIGHBOR
};

enum camera_type_t {
	CAMERA_PERSPECTIVE,
	CAMERA_ORTHOGRAPHIC
};

enum light_shape_t {
	SHAPE_ROUND,
	SHAPE_SQUARE
};

enum data_type_t {
	DATA_TYPE_INVALID,
	DATA_TYPE_INT8,
	DATA_TYPE_INT16,
	DATA_TYPE_INT32,
	DATA_TYPE_INT64,
	DATA_TYPE_UINT8,
	DATA_TYPE_UINT16,
	DATA_TYPE_UINT32,
	DATA_TYPE_UINT64,
	DATA_TYPE_FLOAT16,
	DATA_TYPE_FLOAT32,
	DATA_TYPE_FLOAT64
};

struct animated_transform_params;
struct triangle_mesh_params;
struct shader_params;
struct light_params;
struct particle_stream_interface_data;
struct particle_stream_data;
struct krakatoa_renderer_params;


//
//
// transformation matrices
//
//

/**
 * Transformation matrix.
 * This is a helper class for the renderer. It represents an animated or unanimated transformation matrix.
 */
class CLSEXPORT animated_transform {
private:
	animated_transform_params* m_data;
public:
	/**
	 * Default constructor.
	 * Defaults to the identity matrix. When using the default constructor, the user must use "add_transform" to set the transform.
	 * Users must use this constructor when constructing animated transformations.
	 */
	animated_transform();

	/**
	 * Unanimated matrix constructor from element array.
	 * When using this constructor, the matrix will be unamimated.
	 * @param elements An array of 16 floating point elements representing a 4x4 transformation matrix.
	 */
	animated_transform( const float* elements );

	/**
	 * Unanimated matrix constructor from list of elements.
	 * When using this constructor, the matrix will be unamimated.
	 * @param "eXY" the "X" and "Y" elements of the 4x4 transformation matrix.
	 */
	animated_transform( float e11, float e21, float e31, float e41, float e12, float e22, float e32, float e42, float e13, float e23, float e33, float e43, float e14, float e24, float e34, float e44 ); //unanimated

	~animated_transform();
	animated_transform( const animated_transform& t );
	animated_transform& operator=( const animated_transform& t );
	const animated_transform_params* get_params() const;

	/**
	 * Adds a transformation from element array at a specific shutter time.
	 * Used for constructing animated transformations. This function can be called multiple times for different shutter times.
	 * @param elements An array of 16 floating point elements representing a 4x4 transformation matrix.
	 * @param shutterTime The shutter time in seconds of this matrix.
	 */
	void add_transform( const float* elements, float shutterTime );

	/**
	 * Adds a transformation from element array at a specific shutter time.
	 * Used for constructing animated transformations. This function can be called multiple times for different shutter times.
	 * @param "eXY" the "X" and "Y" elements of the 4x4 transformation matrix.
	 * @param shutterTime The shutter time in seconds of this matrix.
	 */
	void add_transform( float e11, float e21, float e31, float e41, float e12, float e22, float e32, float e42, float e13, float e23, float e33, float e43, float e14, float e24, float e34, float e44, float shutterTime );

	/**
	 * Retrieves a transformation matrix at a given shutter time.
	 * When the transformation matrix is animated, it will use linear interpolation to create a transformation at the given shutter time.
	 * @param outElements The user must pass in a 16 element float array. The resulting transformation matrix will be placed in this array.
	 * @param shutterTime The desired matrix's shutter time in seconds. If it is unanimated, this parameter will not be used.
	 */
	void get_transform( float* outElements, float shutterTime = 0.0f ) const;

	/**
	 * Test to see if this matrix is animated.
	 * @return True if the matrix is animated.
	 */
	bool is_animated() const;
};

//
//
// triangle meshes
//
//

/**
 * Triangle mesh.
 * This is a helper class for the renderer. Triangle meshes are used for matte objects and holdout masks in the renderer.
 * Note that an alternate way to specify matte objects and holdout maskes is to use pre-rendered deep images.
 */
class CLSEXPORT triangle_mesh {
private:
	triangle_mesh_params* m_data;
public:
	/**
	 * Creates an empty mesh.
	 * When using the default constructor, the mesh can be constructed by using the "set_num_vertices", "set_vertex_position" calls, etc.
	 */
	triangle_mesh();

	~triangle_mesh();
	triangle_mesh( const triangle_mesh& t );
	triangle_mesh& operator=( const triangle_mesh& t );

	/**
	 * Sets whether or not this mesh is visible to the camera.
	 * If this is set to true, this mesh will be used as a holdout mask.
	 * @param visibleToCamera Boolean flag.
	 */
	void set_visible_to_camera( bool visibleToCamera );

	/**
	 * Sets whether or not this mesh is visible to the lights in the scene.
	 * If this is set to true, this mesh will be used as a matte object for the scene lights.
	 * @param visibleToLights Boolean flag.
	 */
	void set_visible_to_lights( bool visibleToLights );

	/**
	 * Sets the shutter time that this mesh is represented at.
	 * When the renderer retrieves the mesh, it is offset from this inputted time to the desired shutter time based on the mesh's vertex velocities.
	 * @param shutterTime The shutter time in seconds that this mesh is represented at. Defaults to 0.0.
	 */
	void set_shutter_time( float shutterTime );

	/**
	 * Sets the shutter time that this mesh is represented at.
	 * This is typically only used when doing multi-segment meshes that have changing topology within the renderer's shutter begin and end times.
	 * If the renderer requests a mesh outside of these begin and end times, it will produce an empty mesh.
	 * Example of a two-segement motion blurred mesh. Both mesh1 and mesh2 would be added to the renderer using "add_mesh":
	 *		triangle_mesh mesh1, mesh2;
	 *		triangle_mesh::load_from_file( mesh1, "segment1.xmesh" ); //segment1.xmesh is a mesh with velocities at shutter time -0.05, and is valid from shutter times -0.05 to 0.
	 *		triangle_mesh::load_from_file( mesh2, "segment2.xmesh" ); //segment2.xmesh is a mesh with velocities at shutter time 0, and is valid from shutter times 0 to 0.05.
	 *		mesh1.set_shutter_time( -0.05 );
	 *		mesh1.set_shutter_validity( -0.05, 0.0 );
	 *		mesh2.set_shutter_time( 0.0 );
	 *		mesh2.set_shutter_validity( 0.0, 0.05 );
	 * @param shutterTimeBegin The shutter time in seconds that represents the starting validity range of this mesh. Defaults to -infinity.
	 * @param shutterTimeEnd The shutter time in seconds that represents the ending validity range of this mesh. Defaults to +infinity.
	 */
	void set_shutter_validity( float shutterTimeBegin, float shutterTimeEnd );

	/**
	 * Sets the number of vertices in this mesh.
	 * This function is used when constructing a mesh manually. It must be called before setting vertex data.
	 * @param numVertices The number of vertices in this mesh. Defaults to 0.
	 */
	void set_num_vertices( int numVertices );

	/**
	 * Sets a single vertex position.
	 * @param vertexIndex The vertex index. Must be between 0 and "numVertices" (as set by set_num_vertices).
	 * @param x The "x" component of the vertex position.
	 * @param y The "y" component of the vertex position.
	 * @param z The "z" component of the vertex position.
	 */
	void set_vertex_position( int vertexIndex, float x, float y, float z );

	/**
	 * Sets all the vertex positions with an array.
	 * @param vertexPositionData A float array of length "numVertices" * 3 representing vertex positions. The array order goes as follows: x0,y0,z0,x1,y1,z1,...
	 */
	void set_vertex_position_data( const float* vertexPositionData );

	/**
	 * Sets a single vertex velocity in units per second.
	 * @param vertexIndex The vertex index. Must be between 0 and "numVertices" (as set by set_num_vertices).
	 * @param x The "x" component of the vertex velocity in units per second.
	 * @param y The "y" component of the vertex velocity in units per second.
	 * @param z The "z" component of the vertex velocity in units per second.
	 */
	void set_vertex_velocity( int vertexIndex, float vx, float vy, float vz );

	/**
	 * Sets all the vertex velocities in units per second with an array.
	 * @param vertexVelocityData A float array of length "numVertices" * 3 representing vertex velocities in units per second. The array order goes as follows: vx0,vy0,vz0,vx1,vy1,vz1,...
	 */
	void set_vertex_velocity_data( const float* vertexVelocityData );

	/**
	 * Sets the number of triangles in this mesh.
	 * This function is used when constructing a mesh manually. It must be called before setting face data.
	 * @param numTriangleFaces The number of faces in this mesh. Defaults to 0.
	 */
	void set_num_triangle_faces( int numTriangleFaces );

	/**
	 * Sets a single face's vertices.
	 * @param faceIndex The face index. Must be between 0 and "numTriangleFaces" (as set by set_num_triangle_faces).
	 * @param vert1 The first vertex that makes up this triangle.
	 * @param vert2 The second vertex that makes up this triangle.
	 * @param vert3 The third vertex that makes up this triangle.
	 */
	void set_face( int faceIndex, int vert1, int vert2, int vert3 );

	/**
	 * Sets all the faces with an array.
	 * @param triangleFaceIndices An integer array of length "numTriangleFaces" * 3 representing face vertices. The array order goes as follows: face0vert1,face0vert2,face0vert3,face1vert1,face1vert2,face1vert3,...
	 */
	void set_face_data( const int* triangleFaceIndices );

	/**
	 * Sets this mesh to an empty mesh.
	 */
	void clear();

public:
	/**
	 * Static function that sets a mesh's data based on an "obj" or "xmesh" mesh file.
	 * Note that "obj" files do not include velocity, so to have an animated mesh, use the load_from_files_with_velocities function instead.
	 * @param outTriangleMesh The mesh object to populate.
	 * @param filename The "obj" or "xmesh" filename.
	 */
	static void load_from_file( triangle_mesh& outTriangleMesh, const char* filename );

	/**
	 * Static function that sets a mesh's data based on a pair of "obj" or "xmesh" mesh file.
	 * This function produces a single mesh and populates its vertex velocity based on the change in vertex positions between the two mesh files.
	 * Note that the two meshes passed in must have the same number of vertices.
	 * @param outTriangleMesh The mesh object to populate.
	 * @param filenameBaseMesh An "obj" or "xmesh" filename. This file will be used to generate the mesh's vertices and faces.
	 * @param filenameOffsetMesh An "obj" or "xmesh" filename. This file will be used to generate the mesh's vertex velocities. It must have the same number of vertices as the base mesh.
	 * @param timeBetweenBaseAndOffset The time in seconds between the base mesh and the offset mesh.
	 */
	static void load_from_files_with_velocities( triangle_mesh& outTriangleMesh, const char* filenameBaseMesh, const char* filenameOffsetMesh, float timeBetweenBaseAndOffset );

private:
	void initialize_velocity();
};

//
//
// shader parameters
//
//

/**
 * Shader base class.
 * This is a helper class used to set shading functions within the renderer.
 * Users cannot override this class to produce new shaders. Users can only use the provided subclasses within the renderer.
 */
class CLSEXPORT shader {
protected:
	shader_params* m_data;
public:
	shader();
	~shader();
	shader( const shader& t );
	shader& operator=( const shader& t );
	const shader_params* get_params() const;
	virtual void set_defaults() = 0;
};

/**
 * Isotropic shade function.
 * Light is scattered uniformly independent of the particle, light and camera orientation in space.
 */
class CLSEXPORT shader_isotropic : public shader {
public:
	shader_isotropic();
	virtual void set_defaults();
	//isotropic has no paramters
};

/**
 * Phong phase function.
 * Scatters light dependent on the angle between the light source's direction, the particle normal and the viewing direction according to the Phong specular model.
 * Note that the Phong model describes surface behavior but in Krakatoa every particle, even deep under the surface, will be shaded using this surface shading method.
 * When using low densities, the results might appear unrealistic, producing "shiny volumes".
 * To use this shade function, the particles must have a "Normal" channel.
 * See online documentation for example renders using this function.
 */
class CLSEXPORT shader_phong : public shader {
public:
	shader_phong();
	virtual void set_defaults();
	/// @param value Sets a constant specular power (glossiness) value. Defaults to 10.0.
	void set_specular_power( float value );
	/// @param value Sets a constant specular level value. Defaults to 100.0.
	void set_specular_level( float value );
	/// @param useChannel Use the particle stream's "SpecularPower" channel instead of the constant value. Defaults to false.
	void use_specular_power_channel( bool useChannel );
	/// @param useChannel Use the particle stream's "SpecularLevel" channel instead of the constant value. Defaults to false.
	void use_specular_level_channel( bool useChannel );
};

/**
 * Henyey-Greenstein phase function for ellipsoid-shaped scattering.
 * Scatters light dependent on the angle between the light and the viewing direction.
 * See online documentation for example renders using this function.
 */
class CLSEXPORT shader_henyey_greenstein : public shader {
public:
	shader_henyey_greenstein();
	virtual void set_defaults();
	/// @param value Sets a constant phase eccentricity value (amount of preferential scattering). Can be between -1 and 1. Defaults to 0.0.
	void set_phase_eccentricity( float value );
	/// @param useChannel Use the particle stream's "Eccentricity" channel instead of the constant value. Defaults to false.
	void use_phane_eccentricity_channel( bool useChannel );
};

/**
 * Schlick phase function approximation to the Henyey-Greenstein function.
 * Scatters light dependent on the angle between the light and the viewing direction.
 * See online documentation for example renders using this function.
 */
class CLSEXPORT shader_schlick : public shader {
public:
	shader_schlick();
	virtual void set_defaults();
	/// @param value Sets a constant phase eccentricity value (amount of preferential scattering). Can be between -1 and 1. Defaults to 0.0.
	void set_phase_eccentricity( float value );
	/// @param useChannel Use the particle stream's "Eccentricity" channel instead of the constant value. Defaults to false.
	void use_phane_eccentricity_channel( bool useChannel );
};

/**
 * Kajiya-Kay phase function for rendering of fibers such as hair or cloth.
 * To use this phase function, the particles must have a "Tangent" channel. This function also uses an optional "DiffuseLevel" channel if it exists.
 */
class CLSEXPORT shader_kajiya_kay : public shader {
public:
	shader_kajiya_kay();
	virtual void set_defaults();
	/// @param value Sets a constant specular power (glossiness) value. Defaults to 10.0.
	void set_specular_power( float value );
	/// @param value Sets a constant specular level value. Defaults to 100.0.
	void set_specular_level( float value );
	/// @param useChannel Use the particle stream's "SpecularPower" channel instead of the constant value. Defaults to false.
	void use_specular_power_channel( bool useChannel );
	/// @param useChannel Use the particle stream's "SpecularLevel" channel instead of the constant value. Defaults to false.
	void use_specular_level_channel( bool useChannel );
};

/**
 * Marschner phase function for rendering of hair.
 * See this page for more details: http://renderman.pixar.com/products/whats_renderman/showcase_kingkong.html
 * To use this phase function, the particles must have "Normal" and "Tangent" channels.
 */
class CLSEXPORT shader_marschner : public shader {
public:
	shader_marschner();
	virtual void set_defaults();
	/// @param value Sets a constant specular glossiness. Defaults to 300.0.
	void set_specular_glossiness( float value );
	/// @param value Sets a constant specular level. Defaults to 25.0.
	void set_specular_level( float value );
	/// @param value Sets a constant specular shift. Can be between -1 and 1. Defaults to 0.1.
	void set_specular_shift( float value );
	/// @param value Sets a constant secondary specular glossiness. Defaults to 30.0.
	void set_secondary_specular_glossiness( float value );
	/// @param value Sets a constant secondary specular level. Defaults to 90.0.
	void set_secondary_specular_level( float value );
	/// @param value Sets a constant secondary specular shift. Can be between -1 and 1. Defaults to -0.1.
	void set_secondary_specular_shift( float value );
	/// @param value Sets a constant glint level. Defaults to 400.0.
	void set_glint_level( float value );
	/// @param value Sets a constant glint size. Can be between 0 and 360. Defaults to 0.5.
	void set_glint_size( float value );
	/// @param value Sets a constant glint glossiness. Can be between 0 and 360. Defaults to 10.0.
	void set_glint_glossiness( float value );
	/// @param value Sets a constant diffuse level. Can be between 0 and 100. Defaults to 0.0.
	void set_diffuse_level( float value );
	/// @param useChannel Use the particle stream's "SpecularGlossiness" channel instead of the constant value. Defaults to false.
	void use_specular_glossiness_channel( bool useChannel );
	/// @param useChannel Use the particle stream's "SpecularLevel" channel instead of the constant value. Defaults to false.
	void use_specular_level_channel( bool useChannel );
	/// @param useChannel Use the particle stream's "SpecularShift" channel instead of the constant value. Defaults to false.
	void use_specular_shift_channel( bool useChannel );
	/// @param useChannel Use the particle stream's "Specular2Glossiness" channel instead of the constant value. Defaults to false.
	void use_secondary_specular_glossiness_channel( bool useChannel );
	/// @param useChannel Use the particle stream's "Specular2Level" channel instead of the constant value. Defaults to false.
	void use_secondary_specular_level_channel( bool useChannel );
	/// @param useChannel Use the particle stream's "Specular2Shift" channel instead of the constant value. Defaults to false.
	void use_secondary_specular_shift_channel( bool useChannel );
	/// @param useChannel Use the particle stream's "GlintLevel" channel instead of the constant value. Defaults to false.
	void use_glint_level_channel( bool useChannel );
	/// @param useChannel Use the particle stream's "GlintSize" channel instead of the constant value. Defaults to false.
	void use_glint_size_channel( bool useChannel );
	/// @param useChannel Use the particle stream's "GlintGlossiness" channel instead of the constant value. Defaults to false.
	void use_glint_glossiness_channel( bool useChannel );
	/// @param useChannel Use the particle stream's "DiffuseLevel" channel instead of the constant value. Defaults to false.
	void use_diffuse_level_channel( bool useChannel );
};

//
//
// lights
//
//

/**
 * Light source base class.
 * This is a helper class used to create lights for the renderer.
 * Users cannot override this class to produce new light types. Users can only use the provided subclasses within the renderer.
 */
class CLSEXPORT light {
protected:
	light_params* m_data;
public:
	light();
	~light();
	light( const light& t );
	light& operator=( const light& t );
	const light_params* get_params() const;
	virtual void set_defaults() = 0;

	/// @param name The name of the light. Defaults to "light".
	void set_name( const char* name );
	/// @param r,g,b Specifies the "power" of the light. This is specified as individual r,g,b components. Defaults to [12.5, 12.5, 12.5].
	void set_flux( float r, float g, float b );
	/// @param decayExponent An exponent value representing how the light's intensity reduces over distance. Defaults to 0.0.
	void set_decay_exponent( int decayExponent );
	/// @param enableShadowMap Setting this value to true will enable shadows for this light and will produce shadows from matte objects and use imported deep image maps. Defaults to true.
	void enable_shadow_map( bool enableShadowMap );
	/// @param density Used as a multiplier on the computed shadow values. Defaults to 1.0.
	void set_shadow_density( float density );
	/// @param width Used internally when rendering meshes for matte objects. This is the resolution of the resulting z-depth render. Defaults to 512.
	void set_shadow_map_width( int width );
	/// @param useNearAttenuation Near attenuation is defined as a distance value at which the light "fades in". If this is set, you must also call "set_near_attenuation".
	void use_near_attenuation( bool useNearAttenuation );
	/// @param start,end The starting and ending value at which the light "fades in". The light has no power before "start" and increases linearly to full power by "end".
	void set_near_attenuation( float start, float end );
	/// @param useNearAttenuation Near attenuation is defined as a distance value at which the light "fades out". If this is set, you must also call "set_far_attenuation".
	void use_far_attenuation( bool useFarAttenuation );
	/// @param start,end The starting and ending value at which the light "fades out". The light has full power before "start" and decreases linearly to no power by "end".
	void set_far_attenuation( float start, float end );
};

/**
 * Direct light implementation of the light interface.
 */
class CLSEXPORT direct_light : public light {
public:
	direct_light();
	virtual void set_defaults();

	/// @param lightShape Sets the light to be either square or round in shape. Defaults to krakatoasr::SHAPE_ROUND.
	void set_light_shape( light_shape_t lightShape );
	/// @param lightAspect Only used for square lights. Sets the aspect ratio of width to height of the light rectangle. Defaults to 1.0.
	void set_light_aspect( float lightAspect );
	/// @param innerRadius,outerRadius Sets the radius for the light beam. values between innerRadius and oterRadius are linearly interpolated from full power to no power. Defaults to 43.0 and 45.0 respectively.
	void set_rect_radius( float innerRadius, float outerRadius );
	/// @param inFilename The "dtex" or layered "exr" filename of the light's deep attenuation (shadow) map. Importing external maps can replace having to use meshes for shadow maps.
	void set_attenuation_map_input( const char* inFilename );
	/// @param outFilename The layered "exr" filename for outputting a deep attenuation (shadow) map from this light.
	/// @param numDepthSamples The number of depth layers to produce.
	/// @param sampleSpacing The distance between layers.
	/// @param spacingIsExponential When true, distance between layers grows exponentially instead of linearly.
	void set_attenuation_map_output( const char* outFilename, int numDepthSamples, float sampleSpacing, bool spacingIsExponential );
};

/**
 * Spot light implementation of the light interface.
 */
class CLSEXPORT spot_light : public light {
public:
	spot_light();
	virtual void set_defaults();

	/// @param lightShape Sets the light to be either square or round in shape. Defaults to krakatoasr::SHAPE_ROUND.
	void set_light_shape( light_shape_t lightShape );
	/// @param lightAspect Only used for square lights. Sets the aspect ratio of width to height of the light rectangle. Defaults to 1.0.
	void set_light_aspect( float lightAspect );
	/// @param innerAngle,outerAngle Sets the angle in degress of the spot light's cone. The values between innerAngle and outerAngle are linearly interpolated from full power to no power. Defaults to 43.0 and 45.0 respectively.
	void set_cone_angle( float innerAngle, float outerAngle );
	/// @param inFilename The "dtex" or layered "exr" filename of the light's deep attenuation (shadow) map. Importing external maps can replace having to use meshes for shadow maps.
	void set_attenuation_map_input( const char* inFilename );
	/// @param outFilename The layered "exr" filename for outputting a deep attenuation (shadow) map from this light.
	/// @param numDepthSamples The number of depth layers to produce.
	/// @param sampleSpacing The distance between layers.
	/// @param spacingIsExponential When true, distance between layers grows exponentially instead of linearly.
	void set_attenuation_map_output( const char* outFilename, int numDepthSamples, float sampleSpacing, bool spacingIsExponential );
};

/**
 * Point light implementation of the light interface.
 */
class CLSEXPORT point_light : public light {
public:
	point_light();
	virtual void set_defaults();

	/// @param inFilenameXX The "dtex" filenames of the light's deep attenuation (shadow) maps. Six maps make up the cubemap for this point light. Importing external maps can replace having to render meshes for shadow maps.
	void set_attenuation_cubemap_dtex_input( const char* inFilenameNX, const char* inFilenameNY, const char* inFilenameNZ, const char* inFilenamePX, const char* inFilenamePY, const char* inFilenamePZ );
	/// @param inFilename The layered "exr" filename of the light's deep attenuation (shadow) map. All six cube renders are contained in a single file. Importing external maps can replace having to use meshes for shadow maps.
	void set_attenuation_cubemap_exr_input( const char* inFilename );
	/// @param outFilename The layered "exr" filename for outputting a deep attenuation (shadow) map from this light.
	/// @param numDepthSamples The number of depth layers to produce.
	/// @param sampleSpacing The distance between layers.
	/// @param spacingIsExponential When true, distance between layers grows exponentially instead of linearly.
	void set_attenuation_cubemap_exr_output( const char* outFilename, int numDepthSamples, float sampleSpacing, bool spacingIsExponential );
};


//
//
// user defined custom particle streams
//
//

/**
 * Helper struct for particle_stream_interface. See the particle_stream_interface documentation.
 * Used by the class's get_channel_value and set_channel_value.
 */
struct CLSEXPORT channel_data {
	int byteOffsetInParticle;
	int channelByteSize;
};

/**
 * Stream-style interface class for direct import of particles.
 * Users must create a sub-class and override all the pure virutal member functions.
 *
 * Example sub-class. This stream places 100 particles randomly within a 10x10x10 cube:
 * 
 *	class my_stream : public krakatoasr::particle_stream_interface {
 *		krakatoasr::channel_data m_position;
 *		krakatoasr::channel_data m_velocity;
 *		krakatoasr::channel_data m_density;
 *		int m_count;
 *	public:
 *		my_stream() {
 *			m_position = append_channel( "Position", krakatoasr::DATA_TYPE_FLOAT32, 3 );
 *			m_velocity = append_channel( "Velocity",krakatoasr::DATA_TYPE_FLOAT32, 3 );
 *			m_density = append_channel( "Density", krakatoasr::DATA_TYPE_FLOAT32, 1 );
 *			m_count = 0;
 *		}
 *		virtual krakatoasr::INT64 particle_count() const {
 *			return 100;
 *		}
 *		virtual bool get_particle( void* particleData ) {
 *			//set these values to your own custom particle values
 *			float myPosition[3] = { rand() * 10.0f / RAND_MAX, rand() * 10.0f / RAND_MAX, rand() * 10.0f / RAND_MAX };
 *			float myVelocity[3] = { 0.0f, 0.0f, 0.0f };
 *			float myDensity = 1.0f;
 *			set_channel_value( m_position, particleData, myPosition );
 *			set_channel_value( m_velocity, particleData, myVelocity );
 *			set_channel_value( m_density, particleData, &myDensity );
 *			++m_count;
 *			return ( m_count <= 100 );
 *		}
 *		virtual void close() {}
 *	};
 */
class CLSEXPORT particle_stream_interface {
private:
	particle_stream_interface_data* m_data;

public:
	particle_stream_interface();
	virtual ~particle_stream_interface();
	particle_stream_interface( const particle_stream_interface& c );
	particle_stream_interface& operator=( const particle_stream_interface& c );
	particle_stream_interface_data* get_data() const;

	/**
	 * Adds a channel to the stream.
	 * This is typically called at construction time to define which channels will be available in the stream.
	 * Note that all streams are required to have a "Position" channel. All other channels are optional.
	 * All the channels must be defined prior to adding a stream object to the renderer. See example usage.
	 * @param name The name of the channel. Channel names in Krakatoa have special meaning (for example "Position", "Velocity", "Density", etc.).
	 * @param dataType Specifies the data type of the channel.
	 * @param arity Specifies the arity of the channel. For example, "Position" needs an arity of 3 (xyz), and "Density" needs an arity of 1.
	 * @return A "channel_data" object. This object can be used by get_channel_value and set_channel_value during "get_next_particle. See example usage.
	 */
	channel_data append_channel( const char* name, data_type_t dataType, int arity );

public:
	/**
	 * Defines how many particles this stream will provide. See example usage.
	 * @return The number of particles in the stream, or -1 if unknown.
	 */
	virtual INT64 particle_count() const = 0;

	/**
	 * Gets the next particle in the stream.
	 * The user must use the given memory buffer to set all the channels. This buffer contains all the channels requested by "append_channel".
	 * Channels values for the current particle can be set by using set_channel_value. See example usage.
	 * @param particleData A buffer of memory provided by the system. It is the user's job to populate this buffer with channel data. See example usage.
	 * @return Whether or not there are any particles left in the stream.
	 */
	virtual bool get_next_particle( void* particleData ) = 0;

	/**
	 * Called by the system once the stream is exhaused.
	 * This is a useful function for closing files, or performing cleanup as needed.
	 */
	virtual void close() = 0;

protected:
	/**
	 * Sets a specific channel value within a particle buffer. See example usage.
	 * @param channelData The "accessor" object. This object was provided by "append_channel".
	 * @param particleData The buffer of data for the current particle.
	 * @param inValue A pointer to the channel's new value being set. It should point to data that is of the channel's own data type. See example usage.
	 */
	void set_channel_value( const channel_data& channelData, void* particleData, const void* inValue ) const;

	/**
	 * Strips out a specific channel value from a particle buffer. Its use is very similar to "set_channel_value".
	 * @param channelData The "accessor" object. This object was provided by "append_channel".
	 * @param particleData The buffer of data for the current particle.
	 * @param outValue A pointer to the channel value that was stripped out from particleData. Once retrieved, it should be cast as a pointer to the channel's own data type.
	 */
	void get_channel_value( const channel_data& channelData, const void* particleData, void* outValue ) const;
};

//
//
// particle streams
//
//

/**
 * Class that defines a particle stream for the renderer.
 * All streams provide particles that are at shutter time zero. They are offset to the desired shutter time based on their "Velocity" channel.
 * To create a stream object, users must use one of the static factory functions contained within this class.
 */
class CLSEXPORT particle_stream {
private:
	particle_stream_data* m_data;
public:
	particle_stream();
	~particle_stream();
	particle_stream( const particle_stream& t );
	particle_stream& operator=( const particle_stream& t );
	particle_stream_data* get_data() const;

	/**
	 * Sets a transformation matrix on the particles.
	 * Typically used to move particles which are in object-space into world-space.
	 * This matrix transforms all coordinate channels such as "Position" and "Velocity".
	 * @param tm The matrix. Defaults to the identity matrix.
	 */
	void set_transform( const animated_transform& tm );

public:
	/**
	 * Static factory function to create a particle stream from a "prt" or "bin" file.
	 * @param filename The "prt" or "bin" file name.
	 * @return The resulting particle_stream object which can be added to the renderer.
	 */
	static particle_stream create_from_file( const char* filename );

	/**
	 * Static factory function to create a particle stream from a custom implementation of particle_stream_interface.
	 * @param userStream An object of a sub-class of particle_stream_interface which will provide custom particles to the renderer. Only one particle_stream can be created per userStream instance.
	 * @return The resulting particle_stream object which can be added to the renderer.
	 */
	static particle_stream create_from_particle_stream_interface( particle_stream_interface* userStream );
};

//
//
// progress logger
//
//

/**
 * Interface class to be implemented by the user which allows progress tracking during the renderer.
 * It is useful for providing the user with details of what step Krakatoa is currently performing, and the step's "percent completed" details.
 */
class progress_logger_interface {
public:
	/**
	 * Called by the renderer when it changes its current rendering state. This call should reset the progress to 0.0.
	 * For example, Krakatoa has multiple steps in rendering (sorting, rendering, drawing, etc.). This function is called between these steps.
	 * @param title The name of the step Krakatoa is currently performing.
	 */
	virtual void set_title( const char* title ) = 0;

	/**
	 * Called by the renderer to update the renderer's progress.
	 * Each step will have its own progress and this function will be called to update.
	 * @param progress The fraction completed of the current step. This value will be between 0.0 and 1.0.
	 */
	virtual void set_progress( float progress ) = 0;
};

/// @return A built-in implmentation of the progress_logger_interface which will print logging data to standard out.
FCNEXPORT progress_logger_interface* get_console_progress_logger();

//
//
// frame buffer update
//
//

/**
 * Data layout used by frame_buffer_interface. See "set_frame_buffer" documentation.
 */
struct frame_buffer_pixel_data {
	float r;
	float g;
	float b;
	float r_alpha;
	float g_alpha;
	float b_alpha;
};

/**
 * Interface class to be implemented by the user which allows the user to "see" semi-complete versions of the final rendered image during rendering.
 * It is useful to provide visual feedback of the image during the render.
 */
class frame_buffer_interface {
public:
	/**
	 * Called periodically by the renderer and provides the semi-complete rendered image to the user.
	 * It is also called after the renderer has completed with the final rendered image.
	 * @param width The width of the rendered image.
	 * @param height The height of the rendered image.
	 * @param data An array of length width*height with color and alpha components making up the rendered image.
	 */
	virtual void set_frame_buffer( int width, int height, const frame_buffer_pixel_data* data ) = 0;
};

//
//
// renderer
//
//

/// @level Defines the global logging level of the renderer and how verbose it is. All logging messages go to standard out.
FCNEXPORT void set_global_logging_level( logging_level_t level );
/// @path Defines the path to the DTEX loading dll/so. A DTEX loading library must be accessable via this path if the user is to load DTEX deep images for shadow maps or holdout masks.
FCNEXPORT void set_dtex_library_path( const char* path );

/**
 * This is the main Krakatoa renderer class.
 * It is used to collect all the parameters, lights, particles, meshes, etc. and produce a render.
 */
class CLSEXPORT krakatoa_renderer {
private:
	krakatoa_renderer_params* m_data;
public:
	krakatoa_renderer();
	~krakatoa_renderer();
	krakatoa_renderer( const krakatoa_renderer& t );
	krakatoa_renderer& operator=( const krakatoa_renderer& t );
	const krakatoa_renderer_params* get_params() const;

	//output options

	//void add_output_image_file( const char* filename, output_types_t outputTypes = OUTPUT_RGBA );
	//void add_output_particle_file( const char* filename, bool preLighting = false );
	//void remove_all_outputs();

	/**
	 * THIS FUNCTION IS TEMPORARY, AND WILL BE REPLACED. CURRENTLY ONLY RGBA EXR IMAGES CAN BE OUTPUTTED.
	 * @param filename The output "exr" filename for this render.
	 */
	void set_output_image_file( const char* filename ); 

	//general rendering options

	/**
	 * @param errorOnMissingLicense When set to true, the renderer will thow an error when no license is found. When set to false, the renderer will produce a watermarked image.
	 */
	void set_error_on_missing_license( bool errorOnMissingLicense );
	/**
	 * @param r,g,b Defines the background color used in the output image. Defaults to [0,0,0].
	 */
	void set_background_color( float r, float g, float b );
	/**
	 * The first of two settings required to define the rendered density of each particle.
	 * This value is the coefficient of the density as expressed in scientific notation. Smaller values make the particles appear less dense, while larger values increase the rendered density.
	 * @param densityPerParticle Defines the density coefficient of the particles during the render pass. Defaults to 5.0.
	 */
	void set_density_per_particle( float densityPerParticle );
	/**
	 * The second of two settings required to define the rendered density of each particle.
	 * This value is the exponent of the power of 10, where the density is expressed in scientific notation.
	 * This provides a quick way to make large density adjustments.
	 * @param densityExponent Defines the density exponent of the particles during the render pass. Defaults to -1.0.
	 */
	void set_density_exponent( int densityExponent );
	/**
	 * The first of two settings required to define the lighting pass density of each particle. Typically this is the same as render-pass density.
	 * This value is the coefficient of the density as expressed in scientific notation.
	 * There are two sets of density controls "Lighting Pass Density" and "Final Pass Density".
	 * The Lighting Pass performs the calculation of attenuation maps used for particle shadowing. The Final Pass draws the particles or voxels into the frame buffer.
	 * @param lightingDensityPerParticle Defines the density coefficient of the particles during the lighting pass. Defaults to 5.0.
	 */
	void set_lighting_density_per_particle( float lightingDensityPerParticle );
	/**
	 * The second of two settings required to define the lighting pass density of each particle. Typically this is the same as render-pass density exponent.
	 * This value is the exponent of the power of 10, where the density is expressed in scientific notation.
	 * This provides a quick way to make large density adjustments.
	 * @param lightingDensityExponent Defines the density exponent of the particles during the lighting pass. Defaults to -1.0.
	 */
	void set_lighting_density_exponent( int lightingDensityExponent );
	/**
	 * Sets whether or not the particles will "emit" light without any light source.
	 * The amount of light emitted will be set by the "set_emission_strength" and "set_emission_strength_exponent" functions.
	 * If an "Emission" channel exists within the particles, the renderer will use this channel as a multiplier of emission strength.
	 * @param useEmission Defines whether or not the particles will emit light. Defaults to false.
	 */
	void use_emission( bool useEmission );
	/**
	 * The first of two settings required to define the emission of each particle. Typically this is the same as render-pass density.
	 * This value is the coefficient of the emission as expressed in scientific notation.
	 * @param emissionStrength Defines the emission coefficient of the particles. Defaults to 5.0.
	 */
	void set_emission_strength( float emissionStrength );
	/**
	 * The second of two settings required to define the emission of each particle. Typically this is the same as render-pass density exponent.
	 * This value is the exponent of the power of 10, where the emission is expressed in scientific notation.
	 * This provides a quick way to make large emission adjustments.
	 * @param emissionStrengthExponent Defines the emission exponent of the particles. Defaults to -1.0.
	 */
	void set_emission_strength_exponent( int emissionStrengthExponent );
	/**
	 * The "Absorption" channel of particles can control how much light a particle will absorb.
	 * When using the "Absorption" channel, the user can define per-particle absorption values.
	 * When not using the "Absorption" channel, all particles will have uniform absorbtion value of 1.0.
	 * @param useAbsorptionColor Defines non-uniform per-particle absorption values. Particles must have an "Absorption" channel. Defaults to false.
	 */
	void use_absorption_color( bool useAbsorptionColor );
	/**
	 * Sets Krakatoa's rendering mode.
	 * When set to "particle", every particle will be rendered as a pixel-sized point.
	 * When set to "voxel", every particle will be encoded onto a voxel grid and the grid will be shaded.
	 * @param method The rendering method to use. Defaults to krakatoasr::METHOD_PARTICLE.
	 */
	void set_rendering_method( rendering_method_t method );
	/**
	 * Controls the number of neighbor voxels to filter over when shading a voxel.
	 * A value of 1 means no filtering. Higher values produce fuzzier results at cost of render time. 
	 * Only used in "voxel" rendering mode.
	 * @param radius The filter's radius. Defaults to 1.
	 */
	void set_voxel_filter_radius( int radius );
	/**
	 * Sets the global the resolution of the voxel grid.
	 * Only used in "voxel" rendering mode.
	 * @param voxelSize The size of the voxels. Defaults to 0.5.
	 */
	void set_voxel_size( float voxelSize );
	/**
	 * The Draw Point Filter determines how particles are rendered in Particle Mode when they fall partially into more than one integral pixel in the image plane.
	 * Used during the render pass.
	 * Options are:
	 *   Nearest - The full value of the particle is placed into the nearest pixel of the output image.
	 *   Bilinear - Uses a simple linear equation to split the value of the particle into the affected pixels of the output image.
	 *   Bicubic - Uses a cubic filtering function to average the value of the particle into surrounding pixels of the output image
	 * @param filter The filter style to use. Defaults to krakatoasr::FILTER_BILINEAR
	 * @param filterSize The filter size. Only used for the "Bilinear" filter. Example: A value of 1 affects the nearest 4 pixels. Defaults to 1.
	 */
	void set_draw_point_filter( filter_t filter, int filterSize = 1 );
	/**
	 * This special override forces additive rendering by overwriting certain particle channels under the hood.
	 * The following operations will be performed internally when this option is checked:
	 *   The Color Channel will be copied into the Emission Channel.
	 *   The Color and Absorption channels will be set to black.
	 * @param additiveMode whether or not to use this mode.
	 */
	void set_additive_mode( bool additiveMode );
	/**
	 * The Attenuation lookup filter determines how particle self-shadowing attenuation is evaluated from the attenuation buffer in Particle Mode. 
	 * Used during the lighting pass.
	 * Options are:
	 *   Nearest - The full value of the particle is placed into the nearest pixel of the output image.
	 *   Bilinear - Uses a simple linear equation to split the value of the particle into the affected pixels of the output image.
	 *   Bicubic - Uses a cubic filtering function to average the value of the particle into surrounding pixels of the output image
	 * @param filter The filter style to use. Defaults to krakatoasr::FILTER_BICUBIC
	 * @param filterSize The filter size. Only used for the "Bilinear" filter. Example: A value of 1 affects the nearest 4 pixels. Defaults to 1.
	 */
	void set_attenuation_lookup_filter( filter_t filter, int filterSize = 1 );
	/**
	 * Defines the number of anti-aliasing sub-divisions of the matte renderer of the meshes.
	 * When a particle is partially covered by a matte object, it could be assumed either fully in front of fully behind the object.
	 * By increasing this value, multiple samples will be calculated to determine the actual coverage. 
	 * @param rayDivisions The number of subdivisions of the matte render. 1 will produce a render at the render resolution. 2 will produce a render twice the resolution. Defaults to 1.
	*/
	void set_matte_renderer_supersampling( int subDivisions );
	/**
	 * Sets a deep image "dtex" or multi-layered "exr" file to be used as a holdout mask.
	 * This can be used in place of rendering meshes to produce holdouts.
	 * @param filename The "dtex" or multi-layered "exr" file.
	 */
	void set_deep_matte_filename( const char* filename );

	//shader options

	/**
	 * Sets the phase function Krakatoa will use to render the particles.
	 * THIS FUNCTION IS CURRENTLY UNIMPLEMENTED!
	 * @param shader A subclass object of "shader".
	 */
	void set_shader( const shader* shader );

	//camera options
	void set_camera_tm( const animated_transform& tm );
	void set_camera_type( camera_type_t type );
	void set_camera_perspective_fov( float horizonalFov );
	void set_camera_clipping( float nearClipping, float farClipping );

	//render output
	void set_render_resolution( int xResolution, int yResolution );
	void set_pixel_aspect_ratio( float ratio );

	//motion blur and depth of field
	void enable_motion_blur( bool enableMotionBlur );
	void set_motion_blur( float shutterBegin, float shutterEnd, int numSamples, bool jitteredMotionBlur );
	void enable_depth_of_field( bool enableDepthOfField );
	void set_depth_of_field( float fStop, float focalLength, float focalDistance, float sampleRate );

	//light options
	void add_light( const light* lt, const animated_transform& tm );
	void remove_all_lights();

	//holdout/matte mesh options
	void add_mesh( const triangle_mesh* mesh, const animated_transform& tm ); //UNIMPLEMENTED
	void remove_all_meshes(); //UNIMPLEMENTED

	//add particles to a renderer
	void add_particle_stream( particle_stream stream );
	void remove_all_particle_streams();

	//callbacks
	void set_progress_logger_update( progress_logger_interface* progressLoggerUpdater );
	void set_frame_buffer_update( frame_buffer_interface* frameBufferUpdater );

	//launch the render
	void render();

	//clear the renderer object and reset to defaults
	void reset_renderer();
};

} //namespace krakatoasr

#endif
