from System import *
from System.Diagnostics import *
from System.IO import *
from System.Text import *

from Deadline.Plugins import *
from Deadline.Scripting import *

import hashlib
import os

def GetDeadlinePlugin():
    return DraftPlugin()

def CleanupDeadlinePlugin( deadlinePlugin ):
    deadlinePlugin.Cleanup()

class DraftPlugin (DeadlinePlugin):
    Version = "2.6.7"
    PythonExe = ""
    DraftDirectory = ""
    
    def __init__( self ):
        self.InitializeProcessCallback += self.InitializeProcess
        
        self.StartupDirectoryCallback += self.StartupDirectory
        self.RenderExecutableCallback += self.RenderExecutable
        self.RenderArgumentCallback += self.RenderArgument
        
        self.PreRenderTasksCallback += self.PreRenderTasks
        self.PostRenderTasksCallback += self.PostRenderTasks
        
    def Cleanup(self):
        for stdoutHandler in self.StdoutHandlers:
            del stdoutHandler.HandleCallback
        
        del self.InitializeProcessCallback
        del self.StartupDirectoryCallback
        del self.RenderExecutableCallback
        del self.RenderArgumentCallback
        del self.PreRenderTasksCallback
        del self.PostRenderTasksCallback
    
    def GetFileMD5Hash( self, filename ):
        if ( os.path.isfile( filename ) ):
            fileHash = hashlib.md5()
            file = open( filename, 'r' )
            
            try:
                while True:
                    data = file.read( 1024 )
                    
                    if len(data)  == 0:
                        break
                        
                    fileHash.update( data )
                    
                return fileHash.hexdigest()
            finally:
                file.close()
                
        return None
    

    def DraftAutoUpdate( self, localPath, repoPath ):
        updateNeeded = True
        if ( os.path.exists( localPath ) ):
            localHash = self.GetFileMD5Hash( os.path.join( localPath, "Version" ) )
            repoHash = self.GetFileMD5Hash( os.path.join( repoPath, "Version" ) )
            
            if ( localHash == repoHash ):
                updateNeeded = False
        else:
            self.LogInfo( "Creating local Draft directory..." )
            os.mkdir( localPath )
                
        if ( updateNeeded ):
            self.LogInfo( "Draft upgrade detected, copying updated files..." )
            for filePath in Directory.GetFiles( repoPath ):
                fileName = Path.GetFileName( filePath )
                self.LogInfo( "Copying '%s'..." % fileName )
                File.Copy( filePath, Path.Combine(localPath, fileName), True )
            self.LogInfo( "Draft update completed!" )
    
    
    def InitializeProcess( self ):
        self.PluginType = PluginType.Simple
        self.SingleFramesOnly = False
        self.StdoutHandling = True
        self.ProcessPriority = ProcessPriorityClass.BelowNormal
        
        self.AddStdoutHandlerCallback( ".*Progress: ([0-9]+).*" ).HandleCallback += self.HandleStdoutProgress
        
        draftLocalPath = Path.Combine( self.GetSlaveDirectory(), "Draft" )
        draftRepoPath = Path.Combine( RepositoryUtils.GetRootDirectory(), "draft" )
        
        if not os.path.exists( draftRepoPath ):
            self.FailRender( "ERROR: Draft was not found in the Deadline Repository!" )
        
        if SystemUtils.IsRunningOnMac():
            draftRepoPath = Path.Combine( draftRepoPath, "Mac" )
        else:
            if SystemUtils.IsRunningOnLinux():
                draftRepoPath = Path.Combine( draftRepoPath, "Linux" )
            else:
                draftRepoPath = Path.Combine( draftRepoPath, "Windows" )
            
            if SystemUtils.Is64Bit():
                draftRepoPath = Path.Combine( draftRepoPath, "64bit" )
            else:
                draftRepoPath = Path.Combine( draftRepoPath, "32bit" )
        
        self.DraftAutoUpdate( draftLocalPath, draftRepoPath )
        
        if SystemUtils.IsRunningOnWindows():
            draftLibrary = Path.Combine( draftLocalPath, "Draft.pyd" )
        else:
            draftLibrary = Path.Combine( draftLocalPath, "Draft.so" )
        
        if not os.path.isfile( draftLibrary ):
            self.FailRender( "Could not find local Draft installation." )
        else:
            self.LogInfo( "Found Draft python module at: '%s'" % draftLibrary )
        
        self.DraftDirectory = draftLocalPath
        
        #preserve existing env vars by appending/prepending
        newPythonPath = self.DraftDirectory
        if Environment.GetEnvironmentVariable( "PYTHONPATH" ) != None:
            newPythonPath = newPythonPath + Path.PathSeparator + Environment.GetEnvironmentVariable( "PYTHONPATH" )
            
        newMagickPath =  self.DraftDirectory
        if Environment.GetEnvironmentVariable( "MAGICK_CONFIGURE_PATH" ) != None:
            #give locally set MAGICK path priority over the Draft one
            newMagickPath =  Environment.GetEnvironmentVariable( "MAGICK_CONFIGURE_PATH" ) + Path.PathSeparator + newMagickPath
            
        self.SetProcessEnvironmentVariable( 'PYTHONPATH', newPythonPath )
        self.SetProcessEnvironmentVariable( 'MAGICK_CONFIGURE_PATH', newMagickPath )
        
        if SystemUtils.IsRunningOnLinux():
            newLDPath = self.DraftDirectory
            if Environment.GetEnvironmentVariable( "LD_LIBRARY_PATH" ) != None:
                newLDPath = newLDPath + Path.PathSeparator + Environment.GetEnvironmentVariable( "LD_LIBRARY_PATH" )
                
            self.SetProcessEnvironmentVariable( 'LD_LIBRARY_PATH', newLDPath )
        elif SystemUtils.IsRunningOnMac():
            newDYLDPath = self.DraftDirectory
            if Environment.GetEnvironmentVariable( "DYLD_LIBRARY_PATH" ) != None:
                newDYLDPath = newDYLDPath + Path.PathSeparator + Environment.GetEnvironmentVariable( "DYLD_LIBRARY_PATH" )
    
            self.SetProcessEnvironmentVariable( 'DYLD_LIBRARY_PATH', newDYLDPath )
    
    def RenderExecutable( self ):
        #build up the path parts based on system type & bitness
        if SystemUtils.IsRunningOnWindows():
            pythonExe = "dpython.exe"
        elif SystemUtils.IsRunningOnMac():
            pythonExe = "dpython"
        else:
            pythonExe = "dpython"
        
        pythonPath = Path.Combine( ClientUtils.GetBinDirectory(), pythonExe )
        
        self.LogInfo( "Looking for bundled python at: '%s'" % pythonPath )
        
        if not FileUtils.FileExists( pythonPath ):
            self.FailRender( "Could not find bundled Python executable." )
        
        self.PythonExe = pythonPath
        return self.PythonExe
        
        
    def RenderArgument( self ):
        scriptFile = self.GetPluginInfoEntryWithDefault( "scriptFile", self.GetDataFilename() )
        
        argsList = []
        
        #Get any arguments specified as ScriptArg# in the plugin info file
        i = 0
        while True:
            argument = self.GetPluginInfoEntryWithDefault( "ScriptArg" + str( i ), None )
            
            if ( argument == None ):
                break
            
            #split out the value from the key (if applicable), to process it (for path mapping)
            separator = '='
            tokens = argument.split( separator, 1 )
            value = tokens[-1]
            
            #temporarily strip quotes if there are any
            wasQuoted = False
            if value.startswith( '"' ) and value.endswith( '"' ) and len(value) > 1:
                wasQuoted = True
                value = value[1:-1]
            
            #do some path fixing, in case our value is a path
            value = RepositoryUtils.CheckPathMapping( value, True )
            value = PathUtils.ToPlatformIndependentPath( value )
            
            #put quotes back if we had some
            if wasQuoted:
                value = '"%s"' % value
            
            tokens[-1] = value
            
            #append the re-joined kvp to the arguments
            argsList.append( separator.join( tokens ) )
            i += 1
        
        #Add the repo root as an argument, should already be in a format that the Slave understands
        argsList.append( '%s="%s"' % ( "deadlineRepository", RepositoryUtils.GetRootDirectory() ) )
        
        #Add the current task's start/end frame to the arguments
        argsList.append( '%s=%s' % ( "taskStartFrame", self.GetStartFrame() ) )
        argsList.append( '%s=%s' % ( "taskEndFrame", self.GetEndFrame() ) )
        
        separator = ' '
        arguments = separator.join( argsList )
        
        #check the path of the script itself
        scriptFile = RepositoryUtils.CheckPathMapping( scriptFile, True )
        scriptFile = PathUtils.ToPlatformIndependentPath( scriptFile )
        
        #Get legacy arguments, if any are there
        legacyArgs = self.GetPluginInfoEntryWithDefault( "arguments", "" )
        
        return ( '-u "%s" %s %s' % (scriptFile, arguments, legacyArgs) )
    
    
    def StartupDirectory(self):
        return self.DraftDirectory
    
    def PreRenderTasks(self):
        self.LogInfo( "Draft job starting..." )
        
    def PostRenderTasks(self):
        #check if we're running a post render script (ie, shotgun upload)
        postRenderScript = self.GetPluginInfoEntryWithDefault("postRenderScript", None)
        if ( postRenderScript ) :
            ProcessUtils.SpawnProcess( self.PythonExe, postRenderScript )
        
        self.LogInfo( "Draft job complete!" )
        
    def HandleStdoutError(self):
        self.FailRender( self.GetRegexMatch(0) )
    
    def HandleStdoutProgress(self):
        percentage = float(self.GetRegexMatch(1))
        self.SetProgress( percentage )
