from System.IO import *

from Deadline.Scripting import *
from Deadline.Plugins import *

import os

######################################################################
## This is the function that Deadline calls to get an instance of the
## main DeadlinePlugin class.
######################################################################
def GetDeadlinePlugin():
    return NukePyPlugin()
    return NukePyPlugin()

def CleanupDeadlinePlugin( deadlinePlugin ):
    deadlinePlugin.cleanup()


######################################################################
## This is the main DeadlinePlugin class for the VrayPlugin plugin.
######################################################################
class NukePyPlugin (DeadlinePlugin):
    progress = 0
    currFrame = 0
    tempSceneFilename = ""
    Version = -1.0
    BatchMode = False
    Process = None
    ProcessName = "Nuke"

    def __init__( self ):


        self.InitializeProcessCallback += self.initializeProcess
        self.StartJobCallback += self.StartJob
        self.RenderExecutableCallback += self.renderExecutable
        self.RenderArgumentCallback += self.renderArgument
        self.PreRenderTasksCallback += self.preRenderTasks
        self.PostRenderTasksCallback += self.postRenderTasks
        self.IsSingleFramesOnlyCallback += self.isSingleFramesOnly

        self.TempSceneFilename = None


    def cleanup( self ):
        for stdoutHandler in self.StdoutHandlers:
            del stdoutHandler.HandleCallback

        del self.InitializeProcessCallback
        del self.RenderExecutableCallback
        del self.RenderArgumentCallback
        del self.PreRenderTasksCallback
        del self.PostRenderTasksCallback
        del self.IsSingleFramesOnlyCallback
        del self.StartJobCallback

    def initializeProcess( self ):
        self.SingleFramesOnly = False
        self.StdoutHandling = True
        # -kw- fix per ticket #610729 from deadline for concurrent tasks getting in a stalled state
        self.AsynchronousStdout = False



        # fail on missing frames:
        self.AddStdoutHandlerCallback(".*Read.*Read error: No such file or directory.*").HandleCallback += self.HandleError

        # for when the drive mounts break and we get permiddion problems
        self.AddStdoutHandlerCallback(
            ".*Permission denied:.*").HandleCallback += self.HandleError

        self.AddStdoutHandlerCallback( "Rendering image...: ([0-9]*\.[0-9]*)%.*" ).HandleCallback += self.handleStdoutProgress
        self.AddStdoutHandlerCallback( "Rendering image...: done.*" ).HandleCallback += self.handleStdoutComplete
        self.AddStdoutHandlerCallback( ".*Builder job has completed sucessfully..*").HandleCallback += self.handleStdoutComplete
        self.AddStdoutHandlerCallback( ".*READY FOR INPUT.*").HandleCallback += self.handleStdoutComplete
        self.AddStdoutHandlerCallback( "Starting frame ([0-9]*).*" ).HandleCallback += self.handleStdoutStartFrame
        self.AddStdoutHandlerCallback( "Closing log.*" ).HandleCallback += self.handleStdoutClosing
        # self.AddStdoutHandlerCallback(".*ArithmeticError.*").HandleCallback += self.HandleError
        # self.AddStdoutHandlerCallback(".*AssertionError.*").HandleCallback += self.HandleError
        self.AddStdoutHandlerCallback(".*AttributeError.*").HandleCallback += self.HandleError
        # self.AddStdoutHandlerCallback(".*BufferError.*").HandleCallback += self.HandleError
        # self.AddStdoutHandlerCallback(".*EOFError.*").HandleCallback += self.HandleError
        # self.AddStdoutHandlerCallback(".*EnvironmentError.*").HandleCallback += self.HandleError
        # self.AddStdoutHandlerCallback(".*FloatingPointError.*").HandleCallback += self.HandleError
        self.AddStdoutHandlerCallback(".*IOError.*").HandleCallback += self.HandleError
        # self.AddStdoutHandlerCallback(".*ImportError.*").HandleCallback += self.HandleError
        # self.AddStdoutHandlerCallback(".*IndentationError.*").HandleCallback += self.HandleError
        # self.AddStdoutHandlerCallback(".*IndexError.*").HandleCallback += self.HandleError
        # self.AddStdoutHandlerCallback(".*KeyError.*").HandleCallback += self.HandleError
        # self.AddStdoutHandlerCallback(".*LookupError.*").HandleCallback += self.HandleError
        # self.AddStdoutHandlerCallback(".*MemoryError.*").HandleCallback += self.HandleError
        # self.AddStdoutHandlerCallback(".*NameError.*").HandleCallback += self.HandleError
        # self.AddStdoutHandlerCallback(".*NotImplementedError.*").HandleCallback += self.HandleError
        # self.AddStdoutHandlerCallback(".*OSError.*").HandleCallback += self.HandleError
        # self.AddStdoutHandlerCallback(".*OverflowError.*").HandleCallback += self.HandleError
        # self.AddStdoutHandlerCallback(".*ReferenceError.*").HandleCallback += self.HandleError
        # # self.AddStdoutHandlerCallback(".*RuntimeError.*").HandleCallback += self.HandleError 
        # self.AddStdoutHandlerCallback(".*StandardError.*").HandleCallback += self.HandleError
        # self.AddStdoutHandlerCallback(".*SyntaxError.*").HandleCallback += self.HandleError
        # self.AddStdoutHandlerCallback(".*SystemError.*").HandleCallback += self.HandleError
        # self.AddStdoutHandlerCallback(".*TabError.*").HandleCallback += self.HandleError
        # self.AddStdoutHandlerCallback(".*TypeError.*").HandleCallback += self.HandleError
        # self.AddStdoutHandlerCallback(".*UnboundLocalError.*").HandleCallback += self.HandleError
        # self.AddStdoutHandlerCallback(".*UnicodeDecodeError.*").HandleCallback += self.HandleError
        # self.AddStdoutHandlerCallback(".*UnicodeEncodeError.*").HandleCallback += self.HandleError
        # self.AddStdoutHandlerCallback(".*UnicodeError.*").HandleCallback += self.HandleError
        # self.AddStdoutHandlerCallback(".*UnicodeTranslateError.*").HandleCallback += self.HandleError
        # self.AddStdoutHandlerCallback(".*VMSError.*").HandleCallback += self.HandleError
        # self.AddStdoutHandlerCallback(".*ValueError.*").HandleCallback += self.HandleError
        # self.AddStdoutHandlerCallback(".*WindowsError.*").HandleCallback += self.HandleError
        # self.AddStdoutHandlerCallback(".*ZeroDivisionError.*").HandleCallback += self.HandleError
        # self.AddStdoutHandlerCallback(".*SeverNotFoundError.*").HandleCallback += self.HandleError
        
        
        self.AddStdoutHandlerCallback("error:.*").HandleCallback += self.HandleError

    ## Utility functions
    def WritePython(self, statement):
        self.FlushMonitoredManagedProcessStdout(self.ProcessName)
        self.WriteStdinToMonitoredManagedProcess(self.ProcessName, statement)
        self.WaitForProcess()

    def scrubLibPath(self, envVar):
        ldPaths = Environment.GetEnvironmentVariable(envVar)
        if ldPaths:
            ldPaths = ldPaths.split(":")
            newLDPaths = []
            for ldPath in ldPaths:
                if not re.search("Deadline", ldPath):
                    newLDPaths.append(ldPath)

            if len(newLDPaths):
                newLDPaths = ":".join(newLDPaths)
            else:
                newLDPaths = ""

            self.SetProcessEnvironmentVariable(envVar, newLDPaths)
            del ldPaths
            del newLDPaths

    def scrubLibPaths(self):
        """This solves a library / plugin linking issue with Nuke that occurs
        when Nuke sees the IlmImf library included with Deadline.  It appears that library
        conflicts with the exrWriter and causes it to error out.  This solution
        removes the Deadline library paths from the LD and DYLD library paths
        before Deadline launches Nuke, so Nuke never sees that library.  It seems like
        this fixes the problem. Thanks to Matt Griffith for figuring this one out!"""

        self.LogInfo("Scrubbing the LD and DYLD LIBRARY paths")

        self.scrubLibPath("LD_LIBRARY_PATH")
        self.scrubLibPath("DYLD_LIBRARY_PATH")
        self.scrubLibPath("DYLD_FALLBACK_LIBRARY_PATH")
        self.scrubLibPath("DYLD_FRAMEWORK_PATH")
        self.scrubLibPath("DYLD_FALLBACK_FRAMEWORK_PATH")

    def prepForOFX(self):
        """This solves an issue where Nuke can fail to create the ofxplugincache,
        which causes any script submited to Deadline that uses an OFX plugin to fail.
        Thanks to Matt Griffith for figuring this one out!"""

        self.LogInfo("Prepping OFX cache")
        nukeTempPath = ""

        # temp path for Nuke
        if SystemUtils.IsRunningOnWindows():
            # on windows, nuke temp path is [Temp]\nuke
            nukeTempPath = Path.Combine(Path.GetTempPath(), "nuke")
        else:
            # on *nix, nuke temp path is "/var/tmp/nuke-u" + 'id -u'
            id = PathUtils.GetApplicationPath("id")
            if len(id) == 0:
                self.LogWarning("Could not get path for 'id' process, skipping OFX cache prep")
                return

            startInfo = ProcessStartInfo(id, "-u")
            startInfo.RedirectStandardOutput = True
            startInfo.UseShellExecute = False

            idProcess = Process()
            idProcess.StartInfo = startInfo
            idProcess.Start()
            idProcess.WaitForExit()

            userId = idProcess.StandardOutput.ReadLine()

            idProcess.StandardOutput.Close();
            idProcess.StandardOutput.Dispose();
            idProcess.Close()
            idProcess.Dispose()

            if len(userId) == 0:
                self.LogWarning("Failed to get user id, skipping OFX cache prep")
                return

            nukeTempPath = "/var/tmp/nuke-u" + userId

        self.LogInfo("Checking Nuke temp path: " + nukeTempPath)
        if Directory.Exists(nukeTempPath):
            self.LogInfo("Path already exists")
        else:
            self.LogInfo("Path does not exist, creating it...")
            Directory.CreateDirectory(
                nukeTempPath)  # creating this level of the nuke temp directory seems to be enough to let the ofxplugincache get created -mg

            if Directory.Exists(nukeTempPath):
                self.LogInfo("Path now exists")
            else:
                self.LogWarning("Unable to create path, skipping OFX cache prep")
                return

        self.LogInfo("OFX cache prepped")

    def pathMappingWithFilePermissionFix(self, inFileName, outFileName, stringsToReplace, newStrings):
        RepositoryUtils.CheckPathMappingInFileAndReplace(inFileName, outFileName, stringsToReplace, newStrings)
        if SystemUtils.IsRunningOnLinux() or SystemUtils.IsRunningOnMac():
            os.chmod(outFileName, os.stat(inFileName).st_mode)

    def StartJob(self):
        self.LogInfo("Running Start job")
        self.Version = float(self.GetPluginInfoEntry("Version"))
        # fix old nuke submissions:
        # need to remap to new mapping.
        if self.Version == 9.6:
            self.Version = 9.06
        if self.Version == 7.9:
            self.Version = 7.09


        self.BatchMode = self.GetBooleanPluginInfoEntryWithDefault("BatchMode", True)

        # This fixes a library conflict issue on non-Windows systems.
        if not SystemUtils.IsRunningOnWindows():
            self.scrubLibPaths()

        if self.GetBooleanConfigEntryWithDefault("PrepForOFX", True):
            # Ensure that OFX plugins will work
            try:
                self.prepForOFX()
            except:
                self.LogWarning("Prepping of OFX cache failed")

    def renderExecutable( self ):
        self.Version = float(self.GetPluginInfoEntry("Version"))
        # fix old nuke submissions:
        # need to remap to new mapping.
        if self.Version == 9.6:
            self.Version = 9.06
        if self.Version == 7.9:
            self.Version = 7.09

        nukeExeList = self.GetConfigEntry("RenderExecutable" + str(self.Version).replace(".", "_"))
        nukeExe = FileUtils.SearchFileList(nukeExeList)
        if (nukeExe == ""):
            self.FailRender(
                "Nuke %s render executable could not be found in the semicolon separated list \"%s\". The path to the render executable can be configured from the Plugin Configuration in the Deadline Monitor." % (
                self.Version, nukeExeList))

        return nukeExe

    def renderArgument( self ):
        # Enable verbosity (the '2' option is only available in Nuke 7 and later)
        renderarguments = "-V"

        verbosity_level = self.GetIntegerPluginInfoEntryWithDefault('Verbosity', 2)

        if self.Version >= 7.0:
            renderarguments += " %s" % verbosity_level

        if self.GetBooleanPluginInfoEntryWithDefault("NukeX", False):
            self.LogInfo("Rendering with NukeX")
            renderarguments += " --nukex"

        if self.GetBooleanPluginInfoEntryWithDefault("ContinueOnError", False):
            self.LogInfo(
                "An attempt will be made to render subsequent frames in the range if an error occurs")
            renderarguments += " --cont"

        if self.GetBooleanPluginInfoEntryWithDefault("EnforceRenderOrder", False):
            self.LogInfo("Forcing Nuke to obey the render order of Write nodes")
            renderarguments += " --sro"

        if self.GetBooleanPluginInfoEntryWithDefault("ProxyMode", False):
            self.LogInfo("Rendering using Proxy File Paths")
            renderarguments += " -p"

        # GPU option only supported in Nuke 7 and later.
        if self.Version >= 7.0 and self.GetBooleanPluginInfoEntryWithDefault("UseGpu", False):
            self.LogInfo("Enabling GPU rendering")
            renderarguments += " --gpu"

        thisSlave = self.GetSlaveName().lower()

        interactiveSlaves = self.GetConfigEntryWithDefault("InteractiveSlaves", "").split(',')
        for slave in interactiveSlaves:
            if slave.lower().strip() == thisSlave:
                self.LogInfo(
                    "This slave is in the interactive license list - an interactive license will be used instead of a render license")
                renderarguments += " -i"

        if self.BatchMode:
            renderarguments += " -t"  # start in Terminal mode
        else:
            renderarguments += " -x"  # execute the Nuke script (as opposed to editing it)


        renderarguments += " \"" + self.TempSceneFilename + "\""
        # -kw- if the script is a .py file, then lets see if there are any  command line args given
        command_line_args = self.GetPluginInfoEntryWithDefault("ScriptArguments", "#")
        # only set teh extra command line args if the temp file is a py file and the command line contains something.
        self.LogInfo(" self.TempSceneFilename: %s" % self.TempSceneFilename)
        self.LogInfo(" command_line_args: %s" % command_line_args)
        if self.TempSceneFilename.lower().endswith('.py') and not command_line_args == '#':
            renderarguments += " " + command_line_args

        return renderarguments

    def preRenderTasks( self ):
        self.LogInfo("debug PreRenderTasks")
        sceneFilename = self.GetPluginInfoEntryWithDefault("SceneFile",
                                                                          self.GetDataFilename())
        sceneFilename = RepositoryUtils.CheckPathMapping(sceneFilename)

        # for debuggin, so we can see where the original file is pathed to, without leaving the log
        self.LogInfo("Incoming Filename: %s" % sceneFilename)
        self.TempSceneFilename = None
        if self.GetBooleanConfigEntryWithDefault("EnablePathMapping", True):
            tempSceneDirectory = self.CreateTempDirectory(
                "thread" + str(self.GetThreadNumber()))

            if SystemUtils.IsRunningOnWindows():
                sceneFilename = sceneFilename.replace("/", "\\")
            else:
                sceneFilename = sceneFilename.replace("\\", "/")

            tempSceneFileName = Path.GetFileName(sceneFilename)
            self.TempSceneFilename = Path.Combine(tempSceneDirectory, tempSceneFileName)

            if SystemUtils.IsRunningOnWindows():
                self.TempSceneFilename = self.TempSceneFilename.replace("/", "\\")
                if sceneFilename.startswith("\\") and not sceneFilename.startswith("\\\\"):
                    sceneFilename = "\\" + sceneFilename
                if sceneFilename.startswith("/") and not sceneFilename.startswith("//"):
                    sceneFilename = "/" + sceneFilename
            else:
                self.TempSceneFilename = self.TempSceneFilename.replace("\\", "/")


            # First, replace all TCL escapes ('\]') with '_TCL_ESCAPE_', then replace the '\' path separators with '/', and then swap back in the orignal TCL escapes.
            # This is so that we don't mess up any embedded TCL statements in the output path.
            self.pathMappingWithFilePermissionFix(sceneFilename, self.TempSceneFilename, ("\\[", "\\", "_TCL_ESCAPE_"),
                                                  ("_TCL_ESCAPE_", "/", "\\["))
        else:
            if SystemUtils.IsRunningOnWindows():
                self.TempSceneFilename = sceneFilename.replace("/", "\\")
            else:
                self.TempSceneFilename = sceneFilename.replace("\\", "/")

    def postRenderTasks( self ):
        if self.GetBooleanConfigEntryWithDefault("EnablePathMapping", True):
            File.Delete(self.TempSceneFilename)

    def isSingleFramesOnly( self ):
        # If we are rendering one file per frame, then we can only render one frame at a time.
        separateFilesPerFrame = self.GetBooleanPluginInfoEntryWithDefault( "SeparateFilesPerFrame", False )
        return separateFilesPerFrame

    ######################################################################
    ## Standard Out Handlers
    ######################################################################
    def HandleError(self):

        if (not self.GetBooleanPluginInfoEntryWithDefault("ContinueOnError", False)):
            self.LogInfo(
                'ContinueOnError is set to: %s' % self.GetBooleanPluginInfoEntryWithDefault(
                    "ContinueOnError", False))
            self.FailRender(self.GetRegexMatch(0))
        else:
            self.LogWarning("Skipping error detection as 'Continue On Error' is enabled.")

    def GuruHandleError(self):
        '''
            this is here so we can have errors that we internally deam fatal, that will not get caught on nukes
            noremal continue on error for rendering.


        :return:
        '''

        self.FailRender(self.GetRegexMatch(0))

    def handleStdoutComplete(self):
        # taken from Vray.
        self.LogInfo("marking the progress of the current job to 100% as its done.")
        self.progress = 100

    def HandleProgress(self):
        currFrame = int(self.GetRegexMatch(1))
        totalFrames = int(self.GetRegexMatch(2))
        if totalFrames != 0:
            self.SetProgress((float(currFrame) / float(totalFrames)) * 100.0)
        self.SetStatusMessage(self.GetRegexMatch(0))

    def handleStdoutError( self ):
        self.FailRender( self.GetRegexMatch( 0 ) )
        self.updateProgress()

    def handleStdoutProgress(self):
        self.progress = float( self.GetRegexMatch( 1 ) )
        self.updateProgress()



    def handleStdoutStartFrame( self ):
        self.currFrame = float( self.GetRegexMatch( 1 ) )
        self.SetStatusMessage( "Rendering Frame - " + self.GetRegexMatch( 1 ) )

    def handleStdoutClosing( self ):
        self.SetStatusMessage( "Job Complete" )

    ######################################################################
    ## Helper Functions
    ######################################################################
    def updateProgress( self ):
        startFrame = self.GetStartFrame()
        endFrame = self.GetEndFrame()

        if startFrame == endFrame:
            self.SetProgress( self.progress )
        else:
            overallProgress = ( ( 1.0 / ( endFrame - startFrame + 1 ) ) ) * self.progress
            currentFrameProgress = ( ( ( ( self.currFrame - startFrame ) * 1.0 ) / ( ( ( endFrame - startFrame + 1 ) * 1.0 ) ) ) * 100 )
            self.SetProgress( overallProgress + currentFrameProgress )
