import clr, os

from System import *
from System.Diagnostics import *
from System.IO import *

from Deadline.Plugins import *
from Deadline.Scripting import *

######################################################################
## This is the function that Deadline calls to get an instance of the
## main DeadlinePlugin class.
######################################################################
def GetDeadlinePlugin():
    return VREDPlugin()

def CleanupDeadlinePlugin( deadlinePlugin ):
    deadlinePlugin.Cleanup()

######################################################################
## This is the main DeadlinePlugin class for the VRED plugin.
######################################################################
class VREDPlugin (DeadlinePlugin):

    qualityDict = {"Analytic Low":"VR_QUALITY_ANALYTIC_LOW", "Analytic High":"VR_QUALITY_ANALYTIC_HIGH", "Realistic Low":"VR_QUALITY_REALISTIC_LOW", "Realistic High":"VR_QUALITY_REALISTIC_HIGH", "Raytracing":"VR_QUALITY_RAYTRACING", "NPR":"VR_QUALITY_NPR" }

    def __init__( self ):
        self.InitializeProcessCallback += self.InitializeProcess
        self.RenderExecutableCallback += self.RenderExecutable
        self.RenderArgumentCallback += self.RenderArgument
        self.PreRenderTasksCallback += self.PreRenderTasks
    
    def Cleanup(self):
        for stdoutHandler in self.StdoutHandlers:
            del stdoutHandler.HandleCallback
        
        del self.InitializeProcessCallback
        del self.RenderExecutableCallback
        del self.RenderArgumentCallback
        del self.PreRenderTasksCallback
    
    ## Called by Deadline to initialize the process.
    def InitializeProcess( self ):
        # Set the plugin specific settings.
        self.SingleFramesOnly = False
        self.PluginType = PluginType.Simple
        
        self.ProcessPriority = ProcessPriorityClass.BelowNormal
        self.StdoutHandling = True
        self.PopupHandling = True
        self.PopupMaxChildWindows = 25
    
        self.HandleQtPopups = True
        self.SetEnvironmentVariable("QT_USE_NATIVE_WINDOWS","1")
        
        self.AddPopupHandler( ".*Renderpasses activated.*", "qt_msgbox_buttonbox ")
        
        self.AddPopupIgnorer(".*Autodesk VRED Professional.*")
        self.AddPopupIgnorer(".*Autodesk VRED Server Node.*")
        self.AddPopupIgnorer(".*VREDServerNode*")
        self.AddPopupIgnorer(".*Pipe Configuration Wizard*")
        
        
    def PreRenderTasks( self ):
        sceneFilename = self.GetPluginInfoEntryWithDefault( "SceneFile", "" )
        if sceneFilename != "":
            sceneFilename = RepositoryUtils.CheckPathMapping( sceneFilename )
            if not File.Exists( sceneFilename ):
                self.FailRender( "The VRED project file cannot be found. It wasn't submitted with the job, so it needs to be in a location that is accessible to all Slaves. The VRED project file path is \"" + sceneFilename + "\"." )
    
    def RenderExecutable( self ):
        version = self.GetPluginInfoEntryWithDefault( "Version", "2015" ).strip() #default to empty string (this should match pre-versioning config entries)
        
        vredExeList = self.GetConfigEntry( "RenderExecutable%s" % version )
        vredExe = FileUtils.SearchFileList( vredExeList )
        if( vredExe == "" ):
            self.FailRender( "VRED render executable was not found in the semicolon separated list \"" + vredExeList + "\". The path to the render executable can be configured from the Plugin Configuration in the Deadline Monitor." )

        return vredExe
    
    def RenderArgument( self ):
        sceneFilename = self.GetPluginInfoEntryWithDefault( "SceneFile", self.GetDataFilename() )
        sceneFilename = RepositoryUtils.CheckPathMapping( sceneFilename )
        
        outputFilename = self.GetPluginInfoEntry( "OutputFile" )
        outputFilename = RepositoryUtils.CheckPathMapping( outputFilename )
        
        renderScriptDirectory = self.CreateTempDirectory( "thread" + str(self.GetThreadNumber()) )
        
        renderScript = Path.Combine(renderScriptDirectory,"VRED_RenderScript.py")
        renderScriptDirectory = renderScriptDirectory.replace("\\","\\\\")
        arguments = "-postpython \"import sys;sys.path.append(\\\""+renderScriptDirectory+"\\\");from VRED_RenderScript import *;DeadlineRender()\" "
        #arguments += "-nogui "
        arguments += "\"" + sceneFilename + "\""
        
        writer = StreamWriter( renderScript )
        writer.WriteLine()
        writer.WriteLine( "from vrRenderQueue import *" )
        writer.WriteLine( "from vrController import *" )
        writer.WriteLine( "from vrMovieExport import *")
        writer.WriteLine( "from vrOSGWidget import *")
        writer.WriteLine( "from vrCameraEditor import *")
        writer.WriteLine( "def DeadlineRender():" )
        writer.WriteLine( "\ttry:" )
        
        jobType = self.GetPluginInfoEntryWithDefault( "JobType", "Animation" )
        if jobType == "Render Queue":
            writer.WriteLine( "\t\trunAllRenderJobs()" )
        elif jobType == "Sequencer":
            writer.WriteLine( "\t\trunAllSequences()" )
        else:
            
            view = self.GetPluginInfoEntryWithDefault( "View", "" )
            if not view == "":
                writer.WriteLine("\t\tviewList = getViewpoints()" )
                writer.WriteLine("\t\tif \""+view+"\" in viewList:")
                writer.WriteLine("\t\t\tfor innerView in viewList:")
                writer.WriteLine("\t\t\t\tif not innerView == \""+view+"\":")
                writer.WriteLine("\t\t\t\t\tremoveViewPoint(innerView)")
                writer.WriteLine("\t\t\tjumpViewPoint(\""+view+"\")")
                writer.WriteLine("\t\t\tselectViewPoint(\""+view+"\")")
                writer.WriteLine("\t\t\tselectCamera(\"Perspective\")")
                writer.WriteLine("\t\telse:")
                writer.WriteLine("\t\t\tselectCamera(\""+view+"\")")
        
            animation = self.GetPluginInfoEntryWithDefault( "AnimationClip", "" )
            width = self.GetIntegerPluginInfoEntryWithDefault("RenderWidth", 1920)
            height = self.GetIntegerPluginInfoEntryWithDefault("RenderHeight",1080)
            ssf = self.GetIntegerPluginInfoEntryWithDefault("SuperSamplingFactor",32)
            
            alpha = self.GetBooleanPluginInfoEntryWithDefault("IncludeAlphaChannel",False)
            embed = self.GetBooleanPluginInfoEntryWithDefault("EmbedMetaData",False)
            renderPasses = self.GetBooleanPluginInfoEntryWithDefault("ExportRenderPass",False)
            
            if alpha:
                if outputFilename.endswith(".bmp") or outputFilename.endswith(".jpg") or outputFilename.endswith(".jpeg"):
                    alpha = False
            
            if embed:
                if outputFilename.endswith(".png") or outputFilename.endswith(".jpg") or outputFilename.endswith(".tif"):
                    embedRenderSettings = self.GetBooleanPluginInfoEntryWithDefault("EmbedRenderSettings",False)
                    embedCameraSettings = self.GetBooleanPluginInfoEntryWithDefault("EmbedCameraSettings",False)
                    embedNodeVisibilities = self.GetBooleanPluginInfoEntryWithDefault("EmbedNodeVisibilities",False)
                    embedSwitchNodeStates = self.GetBooleanPluginInfoEntryWithDefault("EmbedSwitchNodeStates",False)
                    embedSwitchMaterialStates = self.GetBooleanPluginInfoEntryWithDefault("EmbedSwitchMaterialStates",False)
                    
                    writer.WriteLine( "\t\tsetSnapshotEmbedMetaData("+embedRenderSettings+", "+embedCameraSettings+", "+embedNodeVisibilities+", "+embedSwitchNodeStates+", "+embedSwitchMaterialStates+")" )
                    
            renderQuality = self.GetPluginInfoEntryWithDefault("RenderQuality","Realistic High")
            writer.WriteLine("\t\tsetRenderQuality("+self.qualityDict[renderQuality]+")")
            
            if renderPasses:
                beauty = str(self.GetBooleanPluginInfoEntryWithDefault("BeautyPass",False))
                diffuseIBL = str(self.GetBooleanPluginInfoEntryWithDefault("DiffuseIBLPass",False))
                diffuseLight = str(self.GetBooleanPluginInfoEntryWithDefault("DiffuseLightPass",False))
                glossyIBL = str(self.GetBooleanPluginInfoEntryWithDefault("GlossyIBLPass",False))
                glossyLight = str(self.GetBooleanPluginInfoEntryWithDefault("GlossyLightPass",False))
                specularReflection = str(self.GetBooleanPluginInfoEntryWithDefault("SpecularReflectionPass",False))
                translucent = str(self.GetBooleanPluginInfoEntryWithDefault("TranslucentPass",False))
                incandescence = str(self.GetBooleanPluginInfoEntryWithDefault("CombinedIncandescencePass",False))
                diffuseIndirect = str(self.GetBooleanPluginInfoEntryWithDefault("DiffuseIndirectPass",False))
                glossyIndirect = str(self.GetBooleanPluginInfoEntryWithDefault("GlossyIndirectPass",False))
                transparencyColor = str(self.GetBooleanPluginInfoEntryWithDefault("TransparencyColorPass",False))
                backgroundColor = str(self.GetBooleanPluginInfoEntryWithDefault("CombinedBackgroundColorPass",False))
                
                writer.WriteLine("\t\tsetCombinedChannelsRenderPasses("+beauty+", "+diffuseIBL+", "+diffuseLight+", "+glossyIBL+", "+glossyLight+", "+specularReflection+", "+translucent+", "+incandescence+", "+diffuseIndirect+", "+glossyIndirect+", "+transparencyColor+", "+backgroundColor+")")
                
                occlusion = str(self.GetBooleanPluginInfoEntryWithDefault("OcclusionPass",False))
                normal = str(self.GetBooleanPluginInfoEntryWithDefault("NormalPass",False))
                depth = str(self.GetBooleanPluginInfoEntryWithDefault("DepthPass",False))
                materialID = str(self.GetBooleanPluginInfoEntryWithDefault("MaterialIDPass",False))
                mask = str(self.GetBooleanPluginInfoEntryWithDefault("MaskPass",False))
                position = str(self.GetBooleanPluginInfoEntryWithDefault("PositionPass",False))
                viewVector = str(self.GetBooleanPluginInfoEntryWithDefault("ViewVectorPass",False))
                
                writer.WriteLine("\t\tsetAuxiliaryChannelsRenderPasses("+occlusion+", "+normal+", "+depth+", "+materialID+", "+mask+", "+position+", "+viewVector+")")
                
                diffuseColor = str(self.GetBooleanPluginInfoEntryWithDefault("DiffuseColorPass",False))
                glossyColor = str(self.GetBooleanPluginInfoEntryWithDefault("GlossyColorPass",False))
                specularColor = str(self.GetBooleanPluginInfoEntryWithDefault("SpecularColorPass",False))
                translucencyColor = str(self.GetBooleanPluginInfoEntryWithDefault("TranslucencyColorPass",False))
                
                writer.WriteLine("\t\tsetMaterialChannelsRenderPasses("+diffuseColor+", "+glossyColor+", "+specularColor+", "+translucencyColor+", "+transparencyColor+")")
                
                IBLDiffuse = str(self.GetBooleanPluginInfoEntryWithDefault("IBLDiffusePass",False))
                IBLGlossy = str(self.GetBooleanPluginInfoEntryWithDefault("IBLGlossyPass",False))
                IBLTranslucency = str(self.GetBooleanPluginInfoEntryWithDefault("IBLTranslucencyPass",False))
                lightsDiffuse = str(self.GetBooleanPluginInfoEntryWithDefault("LightsDiffusePass",False))
                lightsGlossy = str(self.GetBooleanPluginInfoEntryWithDefault("LightsGlossyPass",False))
                lightTranslucency = str(self.GetBooleanPluginInfoEntryWithDefault("LightTranslucencyPass",False))
                indirectDiffuse = str(self.GetBooleanPluginInfoEntryWithDefault("IndirectDiffusePass",False))
                indirectGlossy = str(self.GetBooleanPluginInfoEntryWithDefault("IndirectGlossyPass",False))
                indirectSpecular = str(self.GetBooleanPluginInfoEntryWithDefault("IndirectSpecularPass",False))
                indirectTranslucency = str(self.GetBooleanPluginInfoEntryWithDefault("IndirectTranslucencyPass",False))
                
                writer.WriteLine("\t\tsetIlluminationChannelsRenderPasses("+IBLDiffuse+", "+IBLGlossy+", "+IBLTranslucency+", "+lightsDiffuse+", "+lightsGlossy+", "+lightTranslucency+", "+indirectDiffuse+", "+indirectGlossy+", "+indirectSpecular+", "+indirectTranslucency+")")
                
            bgRed = self.GetFloatPluginInfoEntryWithDefault("BackgroundRed",0.0)
            bgGreen = self.GetFloatPluginInfoEntryWithDefault("BackgroundGreen",0.0)
            bgBlue = self.GetFloatPluginInfoEntryWithDefault("BackgroundBlue",0.0)
            
            alphaPreMult = self.GetBooleanPluginInfoEntryWithDefault("PremultiplyAlpha",False)
            tonemapHdr = self.GetBooleanPluginInfoEntryWithDefault("TonemapHDR",False)
            print "Job Type: "+ jobType
            if jobType == "Animation":
                startFrame = str(self.GetStartFrame())
                endFrame = str(self.GetEndFrame())
                
                writer.WriteLine("\t\trenderAnimation(\""+animation+"\", \""+outputFilename+"\", "+str(width)+", "+str(height)+", 25, "+startFrame+", "+endFrame+", "+str(ssf)+", "+str(alpha)+", "+str(bgRed)+", "+str(bgGreen)+", "+str(bgBlue) +", True, "+str(alphaPreMult)+", "+str(tonemapHdr)+")")
            else:
                dpi = self.GetIntegerPluginInfoEntryWithDefault("DPI", 72)

                writer.WriteLine("\t\tcreateSnapshot(\""+outputFilename+"\", "+str(width)+", "+str(height)+", "+str(ssf)+", "+str(alpha)+", "+str(bgRed)+", "+str(bgGreen)+", "+str(bgBlue)+ ", " + str(dpi) + ", True, False, "+str(alphaPreMult)+", "+str(tonemapHdr)+")")            
                
        writer.WriteLine( "\texcept:" )
        writer.WriteLine( "\t\tcrashVred(1)" )#Crash quietly/no popups
        writer.WriteLine( "\tfinally:" )
        writer.WriteLine( "\t\tterminateVred()" )        
        
        writer.Close()
        
        return arguments
