import re

from System import *
from System.Diagnostics import *
from System.IO import *
from System.Text import *
from System.Text.RegularExpressions import *

from Deadline.Plugins import *
from Deadline.Scripting import *

######################################################################
## This is the function that Deadline calls to get an instance of the
## main DeadlinePlugin class.
######################################################################
def GetDeadlinePlugin():
	return NukePlugin()

######################################################################
## This is the main DeadlinePlugin class for the Nuke plugin.
######################################################################
class NukePlugin (DeadlinePlugin):
	Version = -1
	BatchMode = False
	Process = None
	ProcessName = "Nuke"
	
	## Utility functions
	def WritePython( self, statement ):
		FlushMonitoredManagedProcessStdout( self.ProcessName )
		WriteStdinToMonitoredManagedProcess( self.ProcessName, statement )
		self.WaitForProcess()
	
	def WaitForProcess( self ):
		FlushMonitoredManagedProcessStdout( self.ProcessName )
		WriteStdinToMonitoredManagedProcess( self.ProcessName, self.Process.ReadyForInputCommand() )
		while not self.Process.IsReadyForInput():
			VerifyMonitoredManagedProcess( self.ProcessName )
			FlushMonitoredManagedProcessStdout( self.ProcessName )
			
			blockingDialogMessage = CheckForMonitoredManagedProcessPopups( self.ProcessName )
			if( blockingDialogMessage != "" ):
				FailRender( blockingDialogMessage )
			
			if IsCanceled():
				FailRender( "Received cancel task command" )
			
			Sleep( 100 )
		
		self.Process.ResetReadyForInput()
		
	def scrubLibPaths( self):
		"""This solves a library / plugin linking issue with Nuke that occurs
		when Nuke sees the IlmImf library included with Deadline.  It appears that library
		conflicts with the exrWriter and causes it to error out.  This solution
		removes the Deadline library paths from the LD and DYLD library paths
		before Deadline launches Nuke, so Nuke never sees that library.  It seems like
		this fixes the problem. Thanks to Matt Griffith for figuring this one out!"""
		
		LogInfo("Scrubbing the LD and DYLD LIBRARY paths")
		
		# process LD path
		ldPaths = Environment.GetEnvironmentVariable("LD_LIBRARY_PATH")
		if ldPaths:
			ldPaths = ldPaths.split(":")
			newLDPaths = []
			for ldPath in ldPaths:
				if not re.search("Deadline",ldPath):
					newLDPaths.append(ldPath)
			
			if len(newLDPaths):
				newLDPaths = ":".join(newLDPaths)
			else:
				newLDPaths = ""
			
			Environment.SetEnvironmentVariable("LD_LIBRARY_PATH",newLDPaths)
			del ldPaths
			del newLDPaths
		
		# process DYLD path
		dyldPaths = Environment.GetEnvironmentVariable("DYLD_LIBRARY_PATH")
		if dyldPaths:
			dyldPaths = dyldPaths.split(":")
			newDYLDPaths = []
			for dyldPath in dyldPaths:
				if not re.search("Deadline",dyldPath):
					newDYLDPaths.append(dyldPath)
			
			if len(newDYLDPaths):
				newDYLDPaths = ":".join(newDYLDPaths)
			else:
				newDYLDPaths = ""
			
			Environment.SetEnvironmentVariable("DYLD_LIBRARY_PATH",newDYLDPaths)
			del dyldPaths
			del newDYLDPaths
	
	def prepForOFX(self):
		"""This solves an issue where Nuke can fail to create the ofxplugincache,
		which causes any script submited to Deadline that uses an OFX plugin to fail.
		Thanks to Matt Griffith for figuring this one out!"""
		
		LogInfo("Prepping OFX cache")
		
		# temp path for Nuke
		if IsRunningOnWindows():
			# on windows, nuke temp path is [Temp]\nuke
			nukeTempPath = Path.Combine( Path.GetTempPath(), "nuke" )
		else:
			# on *nix, nuke temp path is "/var/tmp/nuke-u" + 'id -u'
			id = GetApplicationPath( "id" )
			if len(id) == 0:
				LogWarning( "Could not get path for 'id' process, skipping OFX cache prep" )
				return
			
			startInfo = ProcessStartInfo( id, "-u" )
			startInfo.RedirectStandardOutput = True
			startInfo.UseShellExecute = False
			
			idProcess = Process()
			idProcess.StartInfo = startInfo
			idProcess.Start()
			
			if not ProcessUtils.WaitForExit( idProcess, 10000 ):
				LogWarning( "The 'id' process failed to return after 10 seconds, skipping OFX cache prep" )
				return
			
			userId = ""
			if idProcess.StandardOutput != None:
				userId = idProcess.StandardOutput.ReadLine()
			
			if len(userId) == 0:
				LogWarning( "Failed to get user id, skipping OFX cache prep" )
				return
			
			nukeTempPath = "/var/tmp/nuke-u" + userId
		
		LogInfo( "Checking Nuke temp path: " + nukeTempPath)
		if Directory.Exists(nukeTempPath):
			LogInfo( "Path already exists" )
		else:
			LogInfo( "Path does not exist, creating it..." )
			Directory.CreateDirectory(nukeTempPath) #creating this level of the nuke temp directory seems to be enough to let the ofxplugincache get created -mg
			
			if Directory.Exists(nukeTempPath):
				LogInfo( "Path now exists" )
			else:
				LogWarning( "Unable to create path, skipping OFX cache prep" )
				return
		
		LogInfo("OFX cache prepped")
	
	## Called by Deadline to initialize the process.
	def InitializeProcess( self ):
		# Set the plugin specific settings.
		self.SingleFramesOnly = False
		self.PluginType = PluginType.Advanced
	
	def StartJob( self ):
		# This fixes a library conflict issue on non-Windows systems.
		#if not IsRunningOnWindows():
		#	self.scrubLibPaths()
			
		# Ensure that OFX plugins will work
		#self.prepForOFX()
		
		self.Version = int( GetPluginInfoEntry( "Version" ) )
		self.BatchMode = GetBooleanPluginInfoEntryWithDefault( "BatchMode", False )
		
		if self.Version >= 5 and self.BatchMode:
			self.Process = NukeProcess( self.Version, self.BatchMode )
			StartMonitoredManagedProcess( self.ProcessName, self.Process )
	
	def RenderTasks( self ):
		if self.Version >= 5 and self.BatchMode:
			self.Process.ResetFinishedFrameCount()
			
			writeNodeName = GetPluginInfoEntryWithDefault( "WriteNode", "" )
			if writeNodeName != "":
				LogInfo( "Rendering write node " + writeNodeName)
				self.WritePython( "nuke.execute(\"" + writeNodeName + "\", " + str(GetStartFrame()) + "," + str(GetEndFrame()) + ",1)" )
			else:
				#self.WritePython( "nuke.executeMultiple(nuke.allNodes(\"Write\"), ((" + str(GetStartFrame()) + "," + str(GetEndFrame()) + ",1),))" )				
				self.WritePython( "nuke.executeMultiple([i for i in nuke.allNodes(\"Write\") if not i['disable'].getValue()], ((" + str(GetStartFrame()) + "," + str(GetEndFrame()) + ",1),))" )
		else:
			RunManagedProcess( NukeProcess( self.Version, self.BatchMode ) )
	
	def EndJob( self ):
		if self.Version >= 5 and self.BatchMode:
			LogInfo( "Ending Nuke Job" )
			FlushMonitoredManagedProcessStdoutNoHandling( self.ProcessName )
			WriteStdinToMonitoredManagedProcess( self.ProcessName, "quit()" )
			FlushMonitoredManagedProcessStdoutNoHandling( self.ProcessName )
			WaitForMonitoredManagedProcessToExit( self.ProcessName, 2000 )
			ShutdownMonitoredManagedProcess( self.ProcessName )
	
class NukeProcess (ManagedProcess):
	FinishedFrameCount = 0
	TempSceneFilename = ""
	
	Version = -1
	BatchMode = False
	ReadyForInput = False
	
	def __init__( self, version, batchMode ):
		self.Version = version
		self.BatchMode = batchMode
	
	def InitializeProcess( self ):
		# Set the process specific settings.
		self.ProcessPriority = ProcessPriorityClass.BelowNormal
		self.UseProcessTree = True
		self.PopupHandling = True
		self.StdoutHandling = True
		
		# Set the stdout handlers.
		self.AddStdoutHandler( "READY FOR INPUT", self.HandleReadyForInput )
		self.AddStdoutHandler( ".*ERROR:.*", self.HandleError ) 
		self.AddStdoutHandler( ".*Error:.*", self.HandleError ) 
		self.AddStdoutHandler( ".* seconds to execute", self.HandleProgress )
		self.AddStdoutHandler( ".* took [0-9]*\\.[0-9]* seconds", self.HandleProgress )
	
	def PreRenderTasks( self ):
		sceneFilename = CheckPathMapping( GetPluginInfoEntryWithDefault( "SceneFile", GetDataFilename() ) )
		#self.TempSceneFilename = Path.Combine( Path.GetTempPath(), Path.GetFileName( sceneFilename ) )
		self.TempSceneFilename = Path.Combine( Path.GetTempPath(), Path.GetFileNameWithoutExtension( sceneFilename ) + "_thread" + str(GetThreadNumber()) + Path.GetExtension( sceneFilename ) )
		
		#CheckPathMappingInFileAndReplaceSeparator( sceneFilename, self.TempSceneFilename, "\\", "/" )
		File.Copy( sceneFilename, self.TempSceneFilename, True )
		
		self.FinishedFrameCount = 0
		SetStatusMessage( "Rendering frame " + str(GetStartFrame()) )
	
	def PostRenderTasks( self ):
		File.Delete( self.TempSceneFilename )
	
	## Called by Deadline to get the render executable.
	def RenderExecutable( self ):
		build = GetPluginInfoEntryWithDefault( "Build", "None" ).lower()
		
		nukeExeList = GetConfigEntry( "RenderExecutable" + str(self.Version) )
		if self.Version > 4 and build == "32bit":
			nukeExe = SearchFileListFor32Bit( nukeExeList )
			if( nukeExe == "" ):
				FailRender( "32 bit Nuke %d render executable could not be found in the semicolon separated list \"%s\". The path to the render executable can be configured from the Plugin Configuration in the Deadline Monitor." % (self.Version, nukeExeList) )
		elif self.Version > 4 and build == "64bit":
			nukeExe = SearchFileListFor64Bit( nukeExeList )
			if( nukeExe == "" ):
				FailRender( "64 bit Nuke %d render executable could not be found in the semicolon separated list \"%s\". The path to the render executable can be configured from the Plugin Configuration in the Deadline Monitor." % (self.Version, nukeExeList) )
		else:
			nukeExe = SearchFileList( nukeExeList )
			if( nukeExe == "" ):
				FailRender( "Nuke %d render executable could not be found in the semicolon separated list \"%s\". The path to the render executable can be configured from the Plugin Configuration in the Deadline Monitor." % (self.Version, nukeExeList) )
		
		return nukeExe
	
	## Called by Deadline to get the render arguments.
	def RenderArgument( self ):
		# Enable verbosity
		renderarguments = "-V"
		
		thisSlave = Environment.MachineName.lower()
		interactiveSlaves = GetConfigEntryWithDefault( "InteractiveSlaves", "" ).split( ',' )
		for slave in interactiveSlaves:
			if slave.lower().strip() == thisSlave:
				LogInfo( "This slave is in the interactive license list - an interactive license will be used instead of a render license" )
				renderarguments += " -i"
		
		if self.Version >= 5 and self.BatchMode:
			renderarguments += " -t" # start in Terminal mode
		else:
			renderarguments += " -x" # execute the Nuke script (as opposed to editing it)
		
		threads = GetIntegerPluginInfoEntryWithDefault( "Threads", 0 )
		if threads > 0:
			LogInfo( "Using " + str(threads) + " threads for rendering" )
			renderarguments += " -m " + str(threads)
		
		ramUse = GetIntegerPluginInfoEntryWithDefault( "RamUse", 0 )
		if ramUse > 0:
			LogInfo( "Limiting RAM usage to " + str(ramUse) + " MB" )
			renderarguments += " -c " + str(ramUse) + "M"
		
		renderarguments += " \"" + self.TempSceneFilename + "\""
		
		if self.Version < 5 or not self.BatchMode:
			renderarguments += " " + str(GetStartFrame()) + "," + str(GetEndFrame())
		
		return renderarguments
	
	def HandleError( self ):
		FailRender( self.GetRegexMatch( 0 ) )
	
	def HandleProgress( self ):
		self.FinishedFrameCount += 1
		
		startFrame = GetStartFrame()
		endFrame = GetEndFrame()
		totalFrames = endFrame - startFrame + 1
		
		if totalFrames != 0:
			SetProgress( ( float(self.FinishedFrameCount) / float(totalFrames) ) * 100.0 )
		
		currFrame = startFrame + self.FinishedFrameCount
		if currFrame <= endFrame:
			SetStatusMessage( "Rendering frame " + str(currFrame) )
		else:
			SetStatusMessage( "Finished rendering " + str(self.FinishedFrameCount) + " frames" )
	
	def ResetFinishedFrameCount( self ):
		self.FinishedFrameCount = 0
	
	def HandleReadyForInput( self ):
		self.ReadyForInput = True
	
	def IsReadyForInput( self ):
		return self.ReadyForInput
	
	def ResetReadyForInput( self ):
		self.ReadyForInput = False
	
	def ReadyForInputCommand( self ):
		return "print( \"READY FOR INPUT\\n\" )"
	